/*
	SETQRG.CPP	- Marco Savegnago IW3FQG

	Commenti:

        Vedi SETQRG.TXT


	Modifiche:

		28/11/93    - Aggiunto output su porta parallela
					- Aggiunto controllo PTT e CTSS
					- Adattato per compilare con SC++ 6 e BC++ 4

		28/12/95	- Aggiunta lettura diretta degli indirizzi delle porte
					  standard installate nel sistema direttamente
					  dalla pagina 0x40.
					- Aggiunto supporto per FT-912

*/

#include <stdio.h>
#include <dos.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#if defined (__ZTC__) || defined (__SC__)
#include <time.h>   // for msleep
#define     delay(msec)     msleep(msec)
#endif

#define LCR 	comBase + 3
#define MCR 	comBase + 4
#define LSR 	comBase + 5
#define MSR 	comBase + 6

#define TIMER1  040H        /* base I/O Port for 8253 timer */
#define TIMER2  TIMER1+3    /* port to latch count */
#define BITTIME 500		    /* port to latch count */

enum bool	   { false, true };
enum portType  { StdSerial, EmulOnSerial, EmulOnParallel };
enum CTSSType  { EndDec = 0x0A, Enc = 0x4a, Off = 0x8a };
enum PTTStatus { Undefined, Transmit = 0x08, Receive = 0x88 };

typedef unsigned char 	byte;
typedef unsigned short 	word;

/* Globals */
#ifdef	__BETA__
static  const char thisVersion[] = "2.2 BETA";
#else
static  const char thisVersion[] = "2.2";
#endif

static  int     _htoi (char *p);
static	void    uso();
static  byte*   frqString2YaesuCmdString(byte *yaesu, byte *frq);

/*******************************************************************
    La classe TPort e' la base (non astratta) di tutti i tipi di
	porte gestite dal programma.
	Il costruttore richiede un argomento con cui si specifica il
	tipo di porta da usare e al suo interno verra' creata l'istanza
	della porta adatta al tipo scelto runtime.
    Per ulteriori info sui costruttori virtuali vedi articolo di
    Bruce Eckel su C++ Report del Marzo '92.
*******************************************************************/

class TPort
{
	TPort* p;

protected:

	bool valid;
    byte destRadio;

public:

	TPort(word portBase, portType type = StdSerial, byte dest = 1, bool invert = false);
    virtual ~TPort();

	void    catOn();
	void    catOff();
	void    changeFrq(byte* destFrq);
	void    setPTT(PTTStatus status);
	void	setCTSS(CTSSType type);

	virtual bool 	openCom();
	virtual void 	sendCom( byte* string, int len);
	virtual void 	closeCom();
    virtual bool    operator! ();

protected:

	TPort();

};

inline	TPort::~TPort()
{
	delete p;
}

inline	bool  TPort::openCom()
{
	return p->openCom();
}

inline	void TPort::sendCom( byte* string, int len)
{
	p->sendCom(string, len);
}

inline	void TPort::closeCom()
{
	p->closeCom();
}

inline	bool TPort::operator! ()
{
	return bool(valid != true);
}

inline	TPort::TPort()
{
	p = 0;
}

/*******************************************************************

	La classe TStdSerialPort realizza l'output usando il classico
	UART 8250 o derivati

*******************************************************************/

class TStdSerialPort
:
	public TPort
{

public:

	word comBase;
	byte oldLCR;
	byte oldMCR;
	int  oldBaud;


public:
	TStdSerialPort (word base, byte dest);

	virtual	bool 	openCom();
	virtual void    sendCom(byte*, int);
	virtual	void 	closeCom();
};

TStdSerialPort::TStdSerialPort (word base, byte dest)
{
	comBase=base;
	destRadio=dest;
}

/*******************************************************************

	La classe TEmulOnSerialPort modula nei segnali di Handshake
	l'output classico della seriale su TD.

*******************************************************************/

class TEmulOnSerialPort
:
	public TStdSerialPort
{

protected:

	virtual	word getTime();
	virtual	word sendBit(word until, word sigType);
	virtual	void sendByte(unsigned char c);

	byte MARK, SPACE;

public:

	TEmulOnSerialPort(word base, byte dest, bool inverted);

	virtual	bool  	openCom();
    virtual void    sendCom(byte*, int);
    virtual void    closeCom();
};

TEmulOnSerialPort::TEmulOnSerialPort(word base, byte dest, bool inverted)
:
	TStdSerialPort(base, dest)
{
	if(inverted)
	{
	   MARK 	= 1;
	   SPACE	= 0;
	}
	else
	{
	   MARK 	= 0;
	   SPACE	= 1;
	}
}

/*******************************************************************
    La classe TEmulOnParallelPort modula nei segnali di Data Bit
	della parallela l'output classico della seriale su TD.
*******************************************************************/

class TEmulOnParallelPort
:
	public TEmulOnSerialPort
{

public:

	TEmulOnParallelPort(word base, byte dest, bool inverted);

	virtual	bool  	openCom();
	virtual	void 	closeCom();
};


inline	TEmulOnParallelPort::TEmulOnParallelPort(word base, byte dest, bool inverted)
:
	TEmulOnSerialPort(base, dest, inverted)
	{
	}

inline 	bool  	TEmulOnParallelPort::openCom()
{
	return true;
}

inline	void 	TEmulOnParallelPort::closeCom()
{
	/* NOOP */
}


void TPort::changeFrq(byte* destFrq)
{
	byte frequenza[]="00000000";

	byte *s=destFrq;
	byte *d=frequenza;

    /*  Se la frequenza comincia con un "12" (1200 Mhz) sostituisce i 2
        caratteri con una 'C'
    */

    if(*s == '1' && *(s+1) == '2')
	{
		s += 2;
		*d++='C';
	}

    /*  Copia la stringa sul buffer di destinazione */

    while(*s)
		*d++=*s++;

	byte cmd[6];

    /*  la trasforma in BCD */

    frqString2YaesuCmdString(cmd, frequenza);

    /*  Attiva il CAT, spara il comando e lo disattiva */

    catOn();
	sendCom(cmd, 5);
	catOff();
}

/*******************************************************************
	Attiva/disattiva lo stato del PTT
*******************************************************************/

void TPort::setPTT(PTTStatus status)
{
	byte cmd[5];

	cmd[4] = status;

	catOn();
	sendCom(cmd, 5);
	catOff();
}

/*******************************************************************
	Imposta lo stato del CTSS
*******************************************************************/

void TPort::setCTSS(CTSSType type)
{
    byte cmd[5];

    cmd[4] = type;

    catOn();
    sendCom(cmd, 5);
    catOff();
}

/*
**  Attiva CAT
*/

void TPort::catOn()
{
	byte catOn[] = { 0, 0, 0, 0, 0 };
    sendCom(catOn,sizeof(catOn));
}

/*
**  Disattiva CAT
*/

void TPort::catOff()
{
	byte catOff[] = { 0, 0, 0, 0, 0x80 };
    sendCom(catOff,sizeof(catOff));
}

bool TStdSerialPort::openCom()
{

/*
**    Setta DLAB nell'LCR a 1 per poter settare la velocita'
*/

	oldLCR = inp(LCR);
	outp ( LCR, 0x80 );

/*
**    Legge la vecchia velocita', la memorizza in Old_Baud e setta la velocita'
**    della seriale a 4800 baud
*/

    oldBaud = inpw (comBase);
	outpw (comBase, 0x0018);

/*
**    Setta DLAB in LCR a 0 e predispone la porta a funzionare con la
**    configurazione 8N2.
*/

    outp ( LCR , 0x07 );

    oldMCR = inp(MCR);
	outp(MCR,0);

	return true;
}

void TStdSerialPort::closeCom ()
{
	outp  (LCR,0x80);
    outpw (comBase, oldBaud);
    outp  (LCR,oldLCR);
    outp  (MCR,oldMCR);
    outp  (LSR,0);
    outp  (MSR,0);
}

/*  Invia una stringa sulla seriale standard    */

void TStdSerialPort::sendCom(byte *s, int num)
{
    if(destRadio)
    	outp ( MCR, destRadio + 1);

    while (num--)
    {
		outp(comBase , *s++);
		while (!(inp (LSR) & 0x20))
		;

		delay(10);
	}
}


word TEmulOnSerialPort::getTime()
{
	asm {

		MOV AL,0
		OUT TIMER2,AL

		IN  AL,TIMER1
		XCHG    AL,AH
		IN  AL,TIMER1
		XCHG    AL,AH
	}
	return (_AX);
}

word TEmulOnSerialPort::sendBit(word until, word sigType)
{
    word com=comBase;
    byte dest=destRadio;

/*
   Chi sa' come si fa' a usare una variabile membro di una classe in BASM??
   In questo caso la linea che segue in teoria potrebbe venir scritta:

   asm  MOV     DX, this.comBase

   Ma apparentemente non funziona.
*/
asm {

	MOV     DX,com
	IN      AL,DX           /* legge il valore della porta */
	MOV     AH,dest         /* il valore del bitmask */

	MOV 	CX, sigType

	JCXZ 	stop

	OR      AL,AH           /* per forzare a 1 il bit che interessa */

	JMP     outbit
	}

stop:

asm {
	NOT     AH              /* lo nega */
	AND     AL,AH
	}

outbit:

asm {

	OUT     DX,AL
	MOV     BX,until
	}

loop:

asm {
	MOV 	AL,0        	/* legge il valore del timer            */
	OUT 	TIMER2,AL

	IN  	AL,TIMER1
	XCHG    AL,AH
	IN  	AL,TIMER1
	XCHG	AL,AH


	SUB     AX,BX
	JNS     loop

	MOV 	AL,0
	OUT 	TIMER2,AL

	IN  	AL,TIMER1
	XCHG    AL,AH
	IN  	AL,TIMER1
	XCHG    AL,AH
	}

	return (_AX);
}

void TEmulOnSerialPort::sendByte(unsigned char c)
{
	register int databit = 8;		/* lunghezza della word */
	register int stopbit = 2;		/* numero di bit di stop */

	/* Prende il tempo ... col cronometro */

	word from = getTime();

	/* Manda il bit di START */

	sendBit ( from -= BITTIME, SPACE );

	/* Trasmette la word */

	while(databit)
	{
		if (c & 1)
			sendBit ( from -= BITTIME, MARK );

		else
			sendBit ( from -= BITTIME, SPACE );

		c >>= 1;

		databit--;
	}

	/* Manda il bit di STOP */

	sendBit ( from -= BITTIME * stopbit, MARK );
}

bool TEmulOnSerialPort::openCom()
{
    /* Salva lo stato di MCR */

    oldMCR = inp(MCR);
    outp(MCR,0);

	return true;
}

void TEmulOnSerialPort::closeCom ()
{
    outp (MCR,oldMCR);
}

void TEmulOnSerialPort::sendCom(byte *s, int num)
{
	/* disabilita gli interrupt */

	disable();

	/* Invia la stringa byte a byte */

	while(num--)
	{
		sendByte(*s++);
		delay(10);
	}

	/* riabilita gli interrupt */

	enable();
}

/*******************************************************************
	Il costruttore di TPort crea l'istanza del TPort appropriato
	conservando il puntatore al suo interno.
*******************************************************************/

TPort::TPort(word portBase, portType type, byte dest, bool invert )
{
	valid = true;

	switch(type)
	{
		case StdSerial:
			p = new TStdSerialPort(portBase, dest);
			break;

		case EmulOnSerial:
			p = new TEmulOnSerialPort(portBase + 4, dest + 1, invert );
			break;

		case EmulOnParallel:
			p = new TEmulOnParallelPort(portBase, 1 << dest, invert );
			break;

		default:
			valid=false;
			break;
	}
}

/*
**  'Trasforma' la frequenza da stringa a bcd
*/

byte* frqString2YaesuCmdString(byte *yaesu, byte *frq)
{
	int a,b;

	byte* s=frq;
	byte* d=yaesu;

	while(*s)
	{
		a=*s++ - '0';
		b=*s++ - '0';

        /* Patchazza per le frequenze che cominciano con 'c' (1200) */

		if(a==19)
			*d++=0xc0+b;
		else
			*d++=( a * 16 ) + b ;
	}

	*d++=1;
	*d=NULL;

	return (yaesu);
}

int _htoi (char *p)
{
	register int i;
	int ret=0,mult=1;

	i=strlen(p);

	while(i--)
	{
		byte ch = toupper(*(p+i));

		ch -= (isalpha(ch)) ? ('A' - 10) : '0';

		ret += ch * mult;
		mult *= 16;
	}

	return (ret);
}

/*
**   Breve help che viene mostrato quando il programma e' lanciato senza
**   argomenti sulla linea di comando
*/

void uso(void)
{
	 puts
	 (
	 "Imposta QRG a RTX YAESU FT212/FT712/FT912 via Seriale/Parallela\n\n"
	 "USO: SETQRG [/opzioni] <port> <frq>\n\n"
	 "  port = 1|2|3|4 (std COM) 1|2|3 (std LPT) 0 = specifica indirizzo es: 03F8\n"
	 "  frq  = cifre significative della frequenza da impostare\n\n"
	 "    es: 145.000  Mhz = 145           432.6625 Mhz = 4326625\n"
	 "        144.750  Mhz = 14475         435.975  Mhz = 435975\n"
	 "        144.7125 Mhz = 1447125      1297.350  Mhz = 1297350\n\n"
	 "  opzioni:\n"
	 "    I        Inverti MARK e SPACE (Solo in emulPort)\n"
	 "    D<n>     Specifica la radio di destinazione (default 1)\n"
	 "    O<S|E|P> Specifica il modo di output:\n"
	 "             \tS)erial (default) \n"
	 "             \tE)mul su seriale\n"
	 "             \tEmul su P)arallela\n"
	 "    R        Forza RX\n"
	 "    T        Forza TX\n"
	 "\n"
	 " Es: SETQRG /D2 1 14475 Imposta la frq 144.750 Mhz su Radio 2 Std Mode"
	 );
}

int main(int na, char **a)
{
	/*
		L'indirizzo delle porte parallele presenti nel PC si trova nel
		segmento 40h agli offset:

		08	-	LPT1
		0A	-	LPT2
		0C	-	LPT3
		0E	-	LPT4	(Solo in !PS2)
	*/

	//	Quindi faccio un puntatore a una word che mi mappa alla base ...

	word __far * 	std_prtbase;

#if __BORLANDC__ >= 0x0452
		std_prtbase = (word __far*) MK_FP( __Seg0040, 8);
#else
#	ifdef _Windows                  // Windows will use the predefined global selector
		std_prtbase = (word __far*) MAKELONG( 8, &_0040H );
#	else                  			// DOS will point directly to the memory area
		std_prtbase = (word __far*) MK_FP( 0x0040, 8 );
#	endif
#endif

	/*
		L'indirizzo delle porte seriali presenti nel PC si trova nel
		segmento 40h agli offset:

		00	-	COM1
		02	-	COM2
		04	-	COM3
		06	-	COM4
	*/

	//	Quindi faccio un puntatore a una word che mi mappa alla base ...

	word 	__far*	std_combase;

#if __BORLANDC__ >= 0x0452
		std_combase = (word __far*) MK_FP( __Seg0040, 0);
#else
#	ifdef _Windows                  // Windows will use the predefined global selector
		std_combase = (word __far*) MAKELONG( 0, &_0040H );
#	else                  			// DOS will point directly to the memory area
		std_combase = (word __far*) MK_FP( 0x0040, 0 );
#	endif
#endif

	word comBase=0;
	byte *destFrq=NULL;

	byte destRadio		=	1;
	portType outType	=	StdSerial;
	bool	invert      =	false;
	PTTStatus	PTT		=   Undefined;

    printf("SETQRG %s - %s Copyright (c) 19911996 Marco Savegnago IW3FQG\n", thisVersion, __DATE__);

/*
**   Controlla che il numero degli argomenti nella linea di comando siano
**   3 altrimenti stampa le istruzioni.
*/
	if (na < 3)
	{
		uso();
		return false;
	}

	for(int i=1;i<na;i++)
	{
		char *p=a[i];

		if(*p=='-'||*p=='/')
		{
			p++;
			switch(toupper(*p))
			{
				case 'H':
				case '?':
					uso();
					return false;

				case 'I':
					invert = true;
					break;

				case 'D':
					if(isdigit(*(++p)))
							destRadio=*p - '0';

					break;

				case 'O':
					switch(toupper(*(++p)))
					{
						case 'S':
							outType=StdSerial;
							break;

						case 'E':
							outType=EmulOnSerial;
                    		break;

                        case 'P':
							outType=EmulOnParallel;
							break;

                        default:

							printf("Opzione non valida %s\n",a[i]);
							return true;
					}
					break;

				case 'R':
					PTT =	Receive;
					break;

				case 'T':
					PTT =   Transmit;
					break;

				default:
					printf("Opzione non valida %s\n",a[i]);
					return true;

			}
		}
		else if(!comBase)
		{
			if(*p < '0' && *p > '3')
			{
				printf("\n\a?? Porta seriale non valida -> %s\n",p);
				uso();

				return false;
			}
			else
			{
				if(*p=='0')
					comBase=_htoi (p);
				else
					comBase = (outType==EmulOnParallel) ? std_prtbase[(*p - '1')] : std_combase[(*p - '1')];
		   }
		}
		else
			destFrq=(byte *)p;

	}

	if(!comBase && (!destFrq || PTT==Undefined))
	{
		puts("Errore nei parametri\a");
		uso();

		return true;
	}

	TPort thePort(comBase, outType, destRadio, invert);

	if(!thePort)
	{
		puts("Errore inizializzando la porta");

		return true;
	}

	thePort.openCom();

	if(destFrq)
		thePort.changeFrq(destFrq);

	if(PTT!=Undefined)
		thePort.setPTT(PTT);

	thePort.closeCom();

	return false;
}
