/*---------------------------------------------------------------------------
  Name:		dates.c
  Version:	0.5
  Internal:	25
  Date:		31 mar 97
  Author:	Marco Rivellino

  Description:	program main file.

		See the README file for a quick overview of the program
		installation and usage.

---------------------------------------------------------------------------*/
/*
    Copyright (C) 1997  Marco Rivellino & Fabio Menegoni

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* program global constants + definitions */
#include "config.h"

/* date_entry data type */
#include "datentry.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef __TURBOC__
	/* neded for searchpath() */
#include <dir.h>
#else
	/* needed to get the home dir name and file statistic */
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>
#endif


#ifndef __NOEDIT
#ifndef __TURBOC__
extern	void edit_data_file(char *filename, int edit_mode);
#endif
#endif




/*---------------------------------------------------------------------------*/




/*
 *	Open file
 *
 */
FILE *open_file(char *filename, char *openmode)
{
	FILE *fileptr;

	if ((fileptr = fopen(filename, openmode)) == NULL) {
		fprintf(stderr, "Cannot open file: %s\n", filename);
		perror("fopen");
	}
	return fileptr;
}




/*
 *	If pointed char is a comment one return true
 *	comment chars are:   ;  *  #
 *
 */
int is_comment_char(char *string)
{
	return (*string == ';' || *string == '*' || *string == '#');
}




/*
 *	Skip all space or tab chars of src
 *
 */
char *skip_spaces(char *src)
{
	while (*src == ' ' || *src == '\t')
		src++;
	return src;
}




#ifdef __DEBUG
/*
 *	Skip all non space or tab chars of src
 *
 */
char *skip_non_spaces(char *src)
{
	while (*src != ' ' && *src != '\t')
		src++;
	return src;
}
#endif




/*
 *	strip ending LF from src and return ptr to beginning of src
 *
 */
char *strip_lf(char *src)
{
	char *tmp = src;

	while (*tmp && *tmp != '\n')
		tmp++;
	*tmp = '\0';
	return src;
}




/*
 *	store today's date into 'de'
 *
 */
void store_current_date(struct date_entry *de)
{
	time_t curtime = time(NULL);
	struct tm *loctime = localtime(&curtime);

	de->day = loctime->tm_mday;
	de->month = (loctime->tm_mon) + 1;
	de->year = (loctime->tm_year) + 1900;
}




/*
 *	print comment msg of date "stored" if it falls into the stored->look_distance period.
 *	if stored->look_distance == -1 || force_cli_look_distance	=>	use cli_look_distance given
 *		as an agrument in the command line interface (-d option).
 *	if the comment is print =>	return TRUE
 *				else	return FALSE
 *
 */
int print_message(struct date_entry *current, struct date_entry *stored, int distance, int cli_look_distance, int force_cli_look_distance, int compact_output, int look_mode)
{
	char *month_name[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

	int look_distance = (stored->look_distance != -1 && !force_cli_look_distance ? stored->look_distance : cli_look_distance);

	int print_rest_of_message = FALSE;

	/* stored date will recur in at maximum 'look_distance' days */
	if (look_mode >=0 && distance >= 0 && distance <= look_distance) {
		if (distance == 0)
			printf("today ");
		if (distance == 1)
			printf("tomorrow ");
		if (compact_output) {
			if (distance >= 2)
				printf("%d ", distance);
			printf("-> ");
		} else {
			if (distance >= 2)
				printf("in %d days time ", distance);
			if (stored->type == DE_RECU)
				printf("it recurs ");
			else
				printf("it is ");
		}
		print_rest_of_message = TRUE;
	}
	/* stored date has recurred since maximum 'look_distance' days */
	if (look_mode <=0 && distance >= -look_distance && distance < 0) {
		if (distance == -1)
			printf("yesterday ");
		if (compact_output) {
			if (distance <= -2)
				printf("%d ", distance);
			printf("-> ");
		} else {
			if (distance <= -2)
				printf("%d days ago ", -distance);
			if (stored->type == DE_RECU)
				printf("it recurred ");
			else
				printf("it was ");
		}
		print_rest_of_message = TRUE;
	}

	if (print_rest_of_message) {
		if (stored->year) {				/* absolute recurrence (e.g. birthday, anniversary..) */
			printf("%d %s %d: %s",
				stored->day,
				month_name[stored->month - 1],
				stored->year,
				stored->comment);
			if (stored->type == DE_RECU)
				printf(" (%d years)\n", current->year - stored->year);
			else
				printf("\n");
		} else {					/* annual recurrence (e.g. Christmas, Workers' day...) */
			printf("%d %s: %s\n",
				stored->day,
				month_name[stored->month - 1],
				stored->comment);
		}
		return TRUE;
	}
	return FALSE;
}




/*
 *	print the format error message associated to "format_error"
 *
 */
void print_format_error_msg(char *filename, char *format_error_line, int format_error)
{
	fprintf(stderr, "Format error in data file: %s\n"
			"line: %s"
			, filename, format_error_line);
	switch (format_error) {
		case DA_E_DF_BADEOF:	fprintf(stderr, "unespected end of file.\n");
					break;

		case DA_E_DF_NODAY:	fprintf(stderr, "day number is missing.\n");
					break;

		case DA_E_DF_NOTYPE:	fprintf(stderr, "type word is missing.\n");
					break;

		case DA_E_DF_NOLOOKDIST:
					fprintf(stderr, "look distance number is missing.\n");
					break;
	}
}




void print_help_text(void)
{
	printf( "dates v" DA_MAJVERSION "." DA_MINVERSION " Copyright (C)1997  Marco Rivellino & Fabio Menegoni.\n"
#ifdef __DEBUG
		"DEBUG VERSION: "__DATE__"   "__TIME__"\n\n"
#endif
		"This program is distributed under the terms of the GNU GPL 2.0.\n"
		"See the accluded COPYING file for the licence terms and conditions.\n"
		"\n"
		"This program reminds you important dates in the future and the past.\n"
		"\n"
		"Usage:    dates [options] [data_file]\n"
		"\n"
		"options:  -b    backward look mode\n"
		"          -c    compact output format\n"
		"          -dN   set look distance to N days (default = %d, N = 0-%d)\n"
#ifndef __TURBOC__
#ifndef __NOEDIT
		"          -e    edit data file (all date entries)\n"
		"          -ed   edit data file (dates only)\n"
		"          -ei   edit data file (invalid entries only)\n"
		"          -er   edit data file (recurrencies only)\n"
#endif
#endif
		"          -f    forward look mode\n"
		"          -h    show this help\n"
		"          -pN   set page length to N lines (default = %d, N >= 2)\n"
#ifndef __TURBOC__
		"          -s    show dates only once a day\n"
#endif
		"\n"
		"data_file is the file with dates to remember "
#ifdef __TURBOC__
		"(default = DATES.LST in DOS path).\n"
#else
		"(default = ~/.dates.lst).\n"
#endif
		, DEFAULT_LOOK_DISTANCE, MAX_LOOK_DISTANCE, DEFAULT_PAGE_LENGTH);
}




/*---------------------------------------------------------------------------*/
void main(int argc, char *argv[])
{
	int 	look_mode = 0, 				/* -1 = print past dates only 
							   +1 = print future dates only
							    0 = print both future+past dates
							*/
		compact_output = FALSE,			/* use compact output format */
		filename_arg,
		force_cli_look_distance = FALSE,
		edit_mode = EDIT_NONE;			/* edit mode: EDIT_NONE=normal mode, all other EDIT_*=run edit interface (-e* option) */

	int	cli_look_distance = DEFAULT_LOOK_DISTANCE,	/* day distance to look through (backward and forward) */
		page_length = DEFAULT_PAGE_LENGTH;	/* lines to print before pausing */

	FILE	*infile;				/* input file handle */
	char	*filename;				/* input file name */

	int	format_error;				/* data file format error flag */
	char	*format_error_line;			/* data file format error line */

	int	curr_arg,				/* argv counter (shell options) */
		distance,				/* distance between stored and current dates */
		printedlines;				/* printed line cnt */

	struct date_entry	stored,			/* date stored in current line of dates.lst */
				current,		/* current date */
				tmpdate;

#ifndef __TURBOC__
	short int my_midnight = -1;			/* smart option flag */
#endif


	if (argc > 1) {
		if (!STRCASECMP(argv[1], "-h")) {
			print_help_text();
			exit(EXIT_SUCCESS);
		}
		filename_arg = FALSE;
		curr_arg = 1;
		while (curr_arg < argc && !filename_arg) {
			if (argv[curr_arg][0] == '-') {
				switch(argv[curr_arg][1] & 0xDF) {
					case 'B':
						look_mode = -1;
						break;

					case 'C':
						compact_output = TRUE;
						break;

					case 'D':
						if (argv[curr_arg][2])
							cli_look_distance = atoi(&argv[curr_arg][2]);
						else
							cli_look_distance = 0;
						if (cli_look_distance < 0)
							cli_look_distance = 0;
						if (cli_look_distance > MAX_LOOK_DISTANCE)
							cli_look_distance = MAX_LOOK_DISTANCE;
						force_cli_look_distance = TRUE;
						break;
#ifndef __TURBOC__
					case 'E':
						switch(argv[curr_arg][2] & 0xDF) {
							case 'D':	edit_mode = EDIT_DATE;
									break;
							case 'I':	edit_mode = EDIT_INVALID;
									break;
							case 'R':	edit_mode = EDIT_RECU;
									break;
							default:	edit_mode = EDIT_ALL;
									break;
						}
						break;
#endif
					case 'F':
						look_mode = 1;
						break;

					case 'P':
						if (argv[curr_arg][2])
							page_length = atoi(&argv[curr_arg][2]);
						else
							page_length = 2;
						if (page_length < 2)
							page_length = 2;
						break;

#ifndef __TURBOC__
					case 'S':
						if (argv[curr_arg][2])
							my_midnight = atoi(&argv[curr_arg][2]);
						if (my_midnight < 0 || my_midnight > 23)
							my_midnight = MY_MIDNIGHT;
						break;
#endif

					default:
						filename_arg = TRUE;
						curr_arg--;
						break;			/* if no option matches, this is a "-*" filename */
				}
				curr_arg++;
			} else
				filename_arg = TRUE;
		}
	}  /* end of argv[] parsing */


	if (curr_arg <= argc - 1) {
		filename = (char *) malloc(strlen(argv[curr_arg]) + 1);
		strcpy(filename, argv[curr_arg]);
	}
	else
#ifdef __TURBOC__
	{
		char *tmpstr = searchpath("dates.lst");
		filename = (char *) malloc(strlen(tmpstr) + 1);
		strcpy(filename, tmpstr);
							/* DOS: search default data file name
							   in DOS path */
	}
#else
	{						/* Linux: set default name of data file to ~/.dates.lst */
		struct passwd *my_passwd;
		my_passwd = getpwuid(getuid());
		if (!my_passwd) {
			fprintf(stderr, "Couldn't get passwd infos.\n");
			exit(EXIT_FAILURE);
		}
		filename = (char *) malloc(strlen(my_passwd->pw_dir) + 11 + 1);
		strcpy(filename, my_passwd->pw_dir);
		strcat(filename, "/.dates.lst");
	}
#endif

#ifndef __TURBOC__
	if (edit_mode != EDIT_NONE) {
#ifndef __NOEDIT
		edit_data_file(filename, edit_mode);
#endif
		free(filename);
		exit(EXIT_SUCCESS);
	}
#endif

	store_current_date(&current);

	if ((infile = open_file(filename, "r")) == NULL)
		exit(EXIT_FAILURE);

#ifndef __TURBOC__
	if (my_midnight >= 0) {
		struct stat *infile_stat = NULL;	/* infile statistic */
		time_t today5;				/* today at 5 o'clock */

	/* computes diff between today at 5 and infile's atime */
		time_t now = time(NULL);
		struct tm *today5_tm = localtime(&now);
		today5_tm->tm_sec = today5_tm->tm_min = 0;
		today5_tm->tm_hour = my_midnight;
		today5 = mktime(today5_tm);

		infile_stat = (struct stat *)malloc(sizeof(struct stat));
		fstat(fileno(infile), infile_stat);
	
		if (difftime(infile_stat->st_atime, today5) > 0 || difftime(now, today5) < 0)
			exit(2);
	}
#endif

	/* get stored date from file & compute distance with today's date */
	printedlines = 1;
	while (read_date_entry(&stored, infile, &format_error_line, &format_error) && stored.state != DA_E_DE_INVALID) {
		copy_date_entry(&tmpdate, &stored);
		if (stored.month == 0) {
			stored.month = current.month;
			tmpdate.month = current.month;
			tmpdate.year = current.year;
			distance = day_distance(&current, &tmpdate);
			if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode))
				printedlines++;
			else {
				stored.month = current.month + 1;
				tmpdate.month = current.month + 1;
				distance = day_distance(&current, &tmpdate);
				if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode))
					printedlines++;
			}
		} else {
			tmpdate.year = current.year;
			distance = day_distance(&current, &tmpdate);
			if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode))
				printedlines++;
			else {
				tmpdate.year = current.year + 1;
				distance = day_distance(&current, &tmpdate);
				if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode))
					printedlines++;
			}
		}
		if (printedlines % page_length == 0) {
			printf("press ENTER...\n");
			fflush(stdin);
			getchar();
			printedlines = 1;
		}
	} /* while */

	if (format_error != DA_E_DF_OK) {
		print_format_error_msg(filename, format_error_line, format_error);
		exit(EXIT_FAILURE);
	}
	if (stored.state == DA_E_DE_INVALID) {
		fprintf(stderr, "\n**Invalid date entry:\n\n"
				"  %d/%d/%d\n"
				"  %s\n"
				"  %d\n"
				"  %s\n\n"
				"**found in data file: %s\n"
				"**Run \"dates -ei %s\" to correct it.\n\n",
			stored.day, stored.month, stored.year,
			(stored.type == DE_DATE ? "DATE" : "RECU"),
			stored.look_distance,
			stored.comment,
			filename,
			filename);
		exit(EXIT_FAILURE);
	}

	fclose(infile);

	free(filename);
	exit(EXIT_SUCCESS);
}
